
rm(list=ls())

source("packages.R")


################################################################################
## Prepare Variables Tunisia
################################################################################


dat = read.dta13("data_tunisia.dta", convert.factors = FALSE)

dat$muslim = car::recode(dat$Q7_1, "1=1; else=NA")

dat$age = dat$Q14_1

dat$agecat = car::recode(dat$age, "c(17,18,19) = 1;
                                   c(20,21,22,23,24,25,26,27,28,29) = 2;
                                   c(30,31,32,33,34,35,36,37,38,39) = 3; 
                                   c(40,41,42,43,44,45,46,47,48,49) = 4;
                                   c(50,51,52,53,54,55,56,57,58,59) = 5;
                                   c(60,61,62,63,64,65,66,67,68,69) = 6;
                                   c(70,71,72,73,74,75,76,77,78,79) = 6;
                                   c(80,81,82,83,84,85,86,87,88,89) = 6;
                                   c(90,91,92,93,94,95,96,97,98,99) = 6")

dat$female = as.numeric(dat$Gender == 2)

dat$income = car::recode(dat$Q14_7, "8=NA")

dat$edu = dat$Q14_3
dat$college = car::recode(dat$edu, "c(1,2,3,4,5,6,7,8)=0; c(9,10)=1")
dat$highschool = car::recode(dat$edu, "c(1,2,3,4,5)=0; c(6,7,8,9,10)=1")
dat$edu5cat = car::recode(dat$edu, "c(1,2)=1; c(3,4)=2; c(5,6)=3; c(7,8)=4; c(9,10)=5")

dat$polinterest = car::recode(dat$Q2_5, "5=NA")

dat$shalat = car::recode(dat$Q7_6, "5=NA")

dat$socialwelfare = car::recode(dat$Q2_3, "5=NA")

dat.tun = dat

## treatment group
dat.tun$group = NA
dat.tun$group[dat.tun$Q13_1 < 10] = "Secularist - No Info"
dat.tun$group[dat.tun$Q13_2 < 10] = "Secularist - Relaxing"
dat.tun$group[dat.tun$Q13_3 < 10] = "Secularist - Tightening"
dat.tun$group[dat.tun$Q13_4 < 10] = "Islamist - No Info"
dat.tun$group[dat.tun$Q13_5 < 10] = "Islamist - Relaxing"
dat.tun$group[dat.tun$Q13_6 < 10] = "Islamist - Tightening"

dat.tun$dvalcoholexp = NA
dat.tun$dvalcoholexp[dat.tun$group == "Secularist - No Info"] = dat.tun$Q13_1[dat.tun$group == "Secularist - No Info"]
dat.tun$dvalcoholexp[dat.tun$group == "Secularist - Relaxing"] = dat.tun$Q13_2[dat.tun$group == "Secularist - Relaxing"]
dat.tun$dvalcoholexp[dat.tun$group == "Secularist - Tightening"] = dat.tun$Q13_3[dat.tun$group == "Secularist - Tightening"]
dat.tun$dvalcoholexp[dat.tun$group == "Islamist - No Info"] = dat.tun$Q13_4[dat.tun$group == "Islamist - No Info"]
dat.tun$dvalcoholexp[dat.tun$group == "Islamist - Relaxing"] = dat.tun$Q13_5[dat.tun$group == "Islamist - Relaxing"]
dat.tun$dvalcoholexp[dat.tun$group == "Islamist - Tightening"] = dat.tun$Q13_6[dat.tun$group == "Islamist - Tightening"]

dat.tun$dvalcoholexp = car::recode(dat.tun$dvalcoholexp, "1=1; 2=0; else=NA")

## dummy variables
dat.tun$islam = 0
dat.tun$islam[dat.tun$group %in% c("Islamist - No Info", "Islamist - Relaxing", "Islamist - Tightening")] = 1

dat.tun$relax = 0
dat.tun$relax[dat.tun$group %in% c("Secularist - Relaxing", "Islamist - Relaxing")] = 1

dat.tun$tighten = 0
dat.tun$tighten[dat.tun$group %in% c("Secularist - Tightening", "Islamist - Tightening")] = 1


##############################################
## Prepare Variables Indonesia
##############################################

dat = read.dta13("data_jakarta.dta", convert.factors = FALSE)

dat$religion = car::recode(dat$agama, "1='Islam'; 
                                       2='Catholic'; 
                                       3='Protestant';
                                       4='Buddha';
                                       5='Hindu';
                                       6='Konghucu';
                                       7='Others'")
dat$religion = factor(dat$religion, 
                      levels=c("Islam", "Catholic", "Protestant",
                               "Buddha", "Hindu", "Konghucu", "Others"))
dat$muslim = car::recode(dat$agama, "1=1; c(2,3,4,5,6,7)=0; else=NA")


dat$female = dat$genderresponden - 1


dat$edu = dat$q4
dat$college = car::recode(dat$edu, "c(1,2,3,4,5,6,7)=0; c(8,9,10)=1")
dat$highschool = car::recode(dat$edu, "c(1,2,3,4)=0; c(5,6,7,8,9,10)=1")
dat$edu5cat = car::recode(dat$edu, "c(1,2)=1; 3=2; 4=3; 5=4; c(6,7,8,9,10)=5")

dat$age = dat$usia

dat$agecat = car::recode(dat$age, "c(17,18,19) = 1;
                                   c(20,21,22,23,24,25,26,27,28,29) = 2;
                                   c(30,31,32,33,34,35,36,37,38,39) = 3; 
                                   c(40,41,42,43,44,45,46,47,48,49) = 4;
                                   c(50,51,52,53,54,55,56,57,58,59) = 5;
                                   c(60,61,62,63,64,65,66,67,68,69) = 6;
                                   c(70,71,72,73,74,75,76,77,78,79) = 6;
                                   c(80,81,82,83,84,85,86,87,88,89) = 6;
                                   c(90,91,92,93,94,95,96,97,98,99) = 6")


dat$income = dat$q5

dat$shalat = car::recode(dat$q33, "9=NA")

dat$polinterest = car::recode(dat$q35, "9=NA")

dat$socialwelfare = car::recode(dat$q39, "9=NA")


## sample characteristics

sink("Sample Characteristics.txt")
cat("Age Categories =======")
# print(table(dat$agecat))
print(prop.table(table(dat$agecat)))

cat("\n\nFemale ======")
# print(table(dat$female))
print(prop.table(table(dat$female)))

cat("\n\nReligion ========")
# print(table(dat$religion))
print(prop.table(table(dat$religion)))

cat("\n\nEducation ======")
# print(table(dat$edu))
print(prop.table(table(dat$edu)))

sink(NULL)


dat.idn = dat

## treatment group
dat.idn$group = NA
dat.idn$group[(dat.idn$q99 < 10)] = "Islam - Status Quo"
dat.idn$group[(dat.idn$q100 < 10)] = "Pancasila - Status Quo"
dat.idn$group[(dat.idn$q101 < 10)] = "Islam - Relaxing"
dat.idn$group[(dat.idn$q102 < 10)] = "Pancasila - Relaxing"
dat.idn$group[(dat.idn$q103 < 10)] = "Islam - Tightening"
dat.idn$group[(dat.idn$q104 < 10)] = "Pancasila - Tightening"


dat.idn$dvalcoholexp = NA
dat.idn$dvalcoholexp[dat.idn$group == "Islam - Status Quo"] = dat.idn$q99[dat.idn$group == "Islam - Status Quo"]
dat.idn$dvalcoholexp[dat.idn$group == "Pancasila - Status Quo"] = dat.idn$q100[dat.idn$group == "Pancasila - Status Quo"]
dat.idn$dvalcoholexp[dat.idn$group == "Islam - Relaxing"] = dat.idn$q101[dat.idn$group == "Islam - Relaxing"]
dat.idn$dvalcoholexp[dat.idn$group == "Pancasila - Relaxing"] = dat.idn$q102[dat.idn$group == "Pancasila - Relaxing"]
dat.idn$dvalcoholexp[dat.idn$group == "Islam - Tightening"] = dat.idn$q103[dat.idn$group == "Islam - Tightening"]
dat.idn$dvalcoholexp[dat.idn$group == "Pancasila - Tightening"] = dat.idn$q104[dat.idn$group == "Pancasila - Tightening"]

dat.idn$dvalcoholexp = car::recode(dat.idn$dvalcoholexp, "1=1; 2=0; else=NA")

## dummy variables
dat.idn$islam = 0
dat.idn$islam[dat.idn$group %in% c("Islam - Status Quo", "Islam - Relaxing", "Islam - Tightening")] = 1

dat.idn$relax = 0
dat.idn$relax[dat.idn$group %in% c("Islam - Relaxing", "Pancasila - Relaxing")] = 1

dat.idn$tighten = 0
dat.idn$tighten[dat.idn$group %in% c("Islam - Tightening", "Pancasila - Tightening")] = 1


###################################################################################################################
## FUNCTIONS TO PLOT INTERACTION
###################################################################################################################

plot.main.effect <- function(plot.main, title, title.size, legend.title.size, lab.y, ylim) {
  plot.out <-
    plot.main +
    geom_errorbar(aes(ymin=Estimate - 1.96*se, ymax=Estimate + 1.96*se), 
                  width=.1, lwd=.7, position = position_dodge(width = .3)) +
    geom_point(size=2, fill="black", position = position_dodge(width = .3)) +
    theme_bw() + 
    theme(legend.text=element_text(size=legend.title.size), plot.title = element_text(size=title.size)) + 
    ylab(lab.y) +
    xlab("") +
    theme(axis.title.x = element_text(size = 12)) +
    theme(axis.title.y = element_text(size = 12)) +
    theme(axis.text.x = element_text(size = 12)) +
    theme(axis.text.y = element_text(size = 12)) +
    ylim(ylim[1], ylim[2]) +
    ggtitle(title)
  
  return (plot.out)
}


plot.int2 <- function(plot.main, title, obj, obj.label, pos.legend, title.size = 10, legend.title.size, legend.label.size, lab.y, ylim, size.x=15, size.y=15) {
  plot.out <-
    plot.main +
    geom_errorbar(aes(ymin=Estimate - 1.96*se, ymax=Estimate + 1.96*se, col=obj), 
                  width=.1, lwd=1, position = position_dodge(width = .3)) +
    geom_point(aes(col=obj, pch=obj), 
               size=4, fill="black", position = position_dodge(width = .3)) +
    theme_bw() + 
    ylab(lab.y) +
    xlab("") +
    theme(axis.title.x = element_text(size = size.x)) +
    theme(axis.title.y = element_text(size = size.y)) +
    theme(axis.text.x = element_text(size = size.x)) +
    theme(axis.text.y = element_text(size = size.y)) +
    coord_cartesian(ylim=c(ylim[1], ylim[2])) + 
    theme(legend.text=element_text(size=legend.label.size)) + 
    theme(plot.title = element_text(size=title.size)) + 
    guides(pch=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE), 
           col=guide_legend(obj.label, title.theme = element_text(size=legend.title.size), reverse=TRUE)) +
    theme(legend.position = pos.legend) + 
    theme(legend.background = element_rect(fill="white", 
                                           size=.1, linetype="solid")) + 
    ggtitle(title)
  
  return (plot.out)
}

plot.barplot <- function(plot.temp, lab.y, ylim, title, subtitle){

    plot.out <- plot.temp + 
      geom_bar(aes(y = (..count..)/sum(..count..)), color="black") + 
      geom_text(aes(y = ((..count..)/sum(..count..)), 
                label = round((..count..)/sum(..count..),2)), 
                stat = "count", vjust = -0.25) +
      scale_y_continuous(labels = waiver()) +
      labs(y = "Proportion", x = "") +
      theme_bw() + 
      coord_cartesian(ylim=c(0, 0.9)) + 
      ggtitle(title, subtitle = subtitle)
}


####################################################################
## Survey Tunisia
####################################################################

svydat.tun = svydesign(id=~1, data=dat.tun, weights = dat.tun$Weighting)
svydat.tun.muslim = subset(svydat.tun, subset=(muslim==1))

dat.tun.muslim = subset(dat.tun, subset=(muslim==1))

####################################################################
## Survey Indonesia
####################################################################

dat.idn.muslim = subset(dat.idn, subset=muslim==1)


####################################################################
## Plot Support for Healthcare and Social Security
####################################################################

### Tunisia
dat.tun.muslim$socialwelfare_lab = car::recode(dat.tun.muslim$socialwelfare, 
                                      "1='Not at all important'; 
                                       2='Not too important';
                                       3='Somewhat important'; 
                                       4='Very important';
                                       NA='Missing'")
dat.tun.muslim$socialwelfare_lab[dat.tun.muslim$socialwelfare_lab == "Not at all important"] = "Not at all\nimportant"
dat.tun.muslim$socialwelfare_lab[dat.tun.muslim$socialwelfare_lab == "Not too important"] = "Not too\nimportant"
dat.tun.muslim$socialwelfare_lab[dat.tun.muslim$socialwelfare_lab == "Somewhat important"] = "Somewhat\nimportant"
dat.tun.muslim$socialwelfare_lab[dat.tun.muslim$socialwelfare_lab == "Very important"] = "Very\nimportant"

dat.tun.muslim$socialwelfare_lab = factor(dat.tun.muslim$socialwelfare_lab, 
                                          levels=c("Not at all\nimportant", "Not too\nimportant",
                                             "Somewhat\nimportant", "Very\nimportant", "Missing"))

plot.temp = ggplot(dat.tun.muslim, aes(as.factor(socialwelfare_lab)))
barplot.tun = plot.barplot(plot.temp,
                  lab.y = "Proportion", ylim=c(0, 0.9), 
                  title = "Tunisian Sample", 
                  subtitle = "How important is it for the Tunisian government to provide its citizens with affordable\nhealthcare?
")


### Indonesia
dat.idn.muslim$socialwelfare_lab = car::recode(dat.idn.muslim$socialwelfare, 
                                               "1='Not at all important'; 
                                       2='Not too important';
                                       3='Somewhat important'; 
                                       4='Very important';
                                       NA='Missing'")
dat.idn.muslim$socialwelfare_lab[dat.idn.muslim$socialwelfare_lab == "Not at all important"] = "Not at all\nimportant"
dat.idn.muslim$socialwelfare_lab[dat.idn.muslim$socialwelfare_lab == "Not too important"] = "Not too\nimportant"
dat.idn.muslim$socialwelfare_lab[dat.idn.muslim$socialwelfare_lab == "Somewhat important"] = "Somewhat\nimportant"
dat.idn.muslim$socialwelfare_lab[dat.idn.muslim$socialwelfare_lab == "Very important"] = "Very\nimportant"

dat.idn.muslim$socialwelfare_lab = factor(dat.idn.muslim$socialwelfare_lab, 
                                          levels=c("Not at all\nimportant", "Not too\nimportant",
                                                   "Somewhat\nimportant", "Very\nimportant", "Missing"))

plot.temp = ggplot(dat.idn.muslim, aes(as.factor(socialwelfare_lab)))
barplot.idn = plot.barplot(plot.temp,
                               lab.y = "Proportion", ylim=c(0, 0.9), 
                               title = "Indonesian Sample", 
                               subtitle = "In your opinion, how important is the government-run social security and healthcare\nprogram (BPJS)?")

png("Figure 1b.png", width=2000, height=2500, res=300)
barplot.idn / barplot.tun
dev.off()


####################################################################
## Main Effects
####################################################################

### Tunisia
mod.basic.tun = svyglm(dvalcoholexp ~ islam + relax + tighten, design=svydat.tun.muslim, family="quasibinomial")
mod.int.tun =  svyglm(dvalcoholexp ~ islam*relax + islam*tighten, design=svydat.tun.muslim, family="quasibinomial")


### Indonesia
mod.basic.idn = glm(dvalcoholexp ~ islam + relax + tighten, data=dat.idn.muslim, family=binomial(link=logit))
mod.int.idn =  glm(dvalcoholexp ~ islam*relax + islam*tighten, data=dat.idn.muslim, family=binomial(link=logit))



####################################################################
## Plot Main Effects
####################################################################

### Indonesia
out = summary(margins(mod.basic.idn))[, 1:3]
out$factor = c("Islam\n(v. Pancasila)", 
               "Liberal Initiative\n(v. Control)", 
               "Conservative Initiative\n(v. Control)")
out$factor = factor(out$factor,
                    levels = c("Islam\n(v. Pancasila)", 
                               "Liberal Initiative\n(v. Control)", 
                               "Conservative Initiative\n(v. Control)"))
out = out[c(2,3,1),]
colnames(out) = c("Treatment", "Estimate", "se")

plot.temp = ggplot(out, aes(Treatment, Estimate))
plot.basic.idn = plot.main.effect(plot.temp,
                              title = "Marginal Effects of the Treatments (IDN)",
                              title.size = 10,
                              legend.title.size = 15,
                              lab.y = "Estimate",
                              ylim = c(-.4, .4)) + 
  geom_hline(yintercept=0, linetype="dashed") + 
  coord_flip()


### Tunisia
out = summary(margins(mod.basic.tun, design=svydat.tun.muslim))[, 1:3]
out$factor = c("Islamist\n(v. Secularist)", 
               "Liberal Initiative\n(v. Control)", 
               "Conservative Initiative\n(v. Control)")
out$factor = factor(out$factor,
                    levels = c("Islamist\n(v. Secularist)", 
                               "Liberal Initiative\n(v. Control)", 
                               "Conservative Initiative\n(v. Control)"))
colnames(out) = c("Treatment", "Estimate", "se")

plot.temp = ggplot(out, aes(Treatment, Estimate))
plot.basic.tun = plot.main.effect(plot.temp,
                              title = "Marginal Effects of the Treatments (TUN)",
                              title.size = 10,
                              legend.title.size = 15,
                              lab.y = "Estimate",
                              ylim = c(-.4, .4)) + 
  geom_hline(yintercept=0, linetype="dashed") + 
  coord_flip()

png("Figure 2.png", width=3000, height=2000, res=300)
plot.basic.idn + plot.basic.tun
dev.off()


####################################################################
## Interaction Effects with High School
####################################################################

mod.int.highschool.tun = svyglm(dvalcoholexp ~ islam*highschool + relax*highschool + tighten*highschool, 
                            design=svydat.tun.muslim, family="quasibinomial")
mod.int3.highschool.tun = svyglm(dvalcoholexp ~ islam*relax*highschool + islam*tighten*highschool, 
                             design=svydat.tun.muslim, family="quasibinomial")


mod.int.highschool.idn = glm(dvalcoholexp ~ islam*highschool + relax*highschool + tighten*highschool, 
                             data=dat.idn.muslim, family=binomial(link="logit"))
mod.int3.highschool.idn = glm(dvalcoholexp ~ islam*relax*highschool + islam*tighten*highschool, 
                              data=dat.idn.muslim, family=binomial(link="logit"))





####################################################################
## Plot Interaction Effects with High School
####################################################################

### Tunisia #################
out = summary(margins(mod.int.highschool.tun, at = list(highschool=c(0, 1)), design=svydat.tun.muslim))[-c(1,2), 1:4]
colnames(out) = c("Treatment", "Education", "Estimate", "se")
out$Education = c("No High School", "High School")
out$Education = factor(out$Education, levels = c("No High School", "High School"))

out$Treatment = c("Islamist", "Islamist",
                  "Liberal\nInitiative", "Liberal\nInitiative", 
                  "Conservative\nInitiative", "Conservative\nInitiative")
out$Treatment = factor(out$Treatment,
                       levels = c("Islamist", 
                                  "Liberal\nInitiative", 
                                  "Conservative\nInitiative"))
plot.temp = ggplot(out, aes(Treatment, Estimate))
plot.int.highschool.tun = plot.int2(plot.temp, 
                                title="Marginal Effects of the Treatments by Education (TUN)",
                                obj=out$Education,
                                obj.label = "Education",
                                pos.legend = c(.20, .10),
                                legend.title.size = 8,
                                legend.label.size = 8,
                                lab.y = "Estimate",
                                ylim = c(-.9, .2),
                                size.x = 10,
                                size.y = 10,
                                title.size = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()


### Indonesia #################
out = summary(margins(mod.int.highschool.idn, at = list(highschool=c(0, 1)), design=svydat.idn.muslim))[-c(1,2), 1:4]
colnames(out) = c("Treatment", "Education", "Estimate", "se")
out$Education = c("No High School", "High School")
out$Education = factor(out$Education, levels = c("No High School", "High School"))

out$Treatment = c("Islamist", "Islamist",
                  "Liberal\nInitiative", "Liberal\nInitiative", 
                  "Conservative\nInitiative", "Conservative\nInitiative")
out$Treatment = factor(out$Treatment,
                       levels = c("Islamist", 
                                  "Liberal\nInitiative", 
                                  "Conservative\nInitiative"))
plot.temp = ggplot(out, aes(Treatment, Estimate))
plot.int.highschool.idn = plot.int2(plot.temp, 
                                    title="Marginal Effects of the Treatments by Education (IDN)",
                                    obj=out$Education,
                                    obj.label = "Education",
                                    pos.legend = c(.20, .10),
                                    legend.title.size = 8,
                                    legend.label.size = 8,
                                    lab.y = "Estimate",
                                    ylim = c(-.9, .2),
                                    size.x = 10,
                                    size.y = 10,
                                    title.size = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()

png("Figure 3.png", width=3000, height=2000, res=300)
plot.int.highschool.idn + plot.int.highschool.tun
dev.off()


####################################################################
## Robustness Tests
####################################################################

### Main Effects and Interaction between Treatments with Covariates
cov.basic.tun = svyglm(dvalcoholexp ~ islam + relax + tighten +
                         female + agecat+ income + polinterest + shalat + socialwelfare,
                       design=svydat.tun.muslim, family="quasibinomial")
cov.int.tun =  svyglm(dvalcoholexp ~ islam*relax + islam*tighten +
                        female + agecat+ income + polinterest + shalat + socialwelfare, 
                      design=svydat.tun.muslim, family="quasibinomial")


cov.basic.idn = glm(dvalcoholexp ~ islam + relax + tighten +
                      female + agecat + income + polinterest + shalat + socialwelfare,
                    data=dat.idn.muslim, family=binomial(link=logit))
cov.int.idn =  glm(dvalcoholexp ~ islam*relax + islam*tighten +
                     female + agecat + income + polinterest + shalat + socialwelfare,
                   data=dat.idn.muslim, family=binomial(link=logit))


### Interaction with High School with Covariates
cov.int.highschool.tun = svyglm(dvalcoholexp ~ female + agecat + income + polinterest + shalat + socialwelfare +
                                  islam*highschool + relax*highschool + tighten*highschool, 
                                design=svydat.tun.muslim, family="quasibinomial")

cov.int.highschool.idn = glm(dvalcoholexp ~ female + agecat + income + polinterest + shalat + socialwelfare + 
                               islam*highschool + relax*highschool + tighten*highschool, 
                             data = dat.idn.muslim, family=binomial(link="logit"))


### Continuous education variable
mod.int.edu.tun = svyglm(dvalcoholexp ~ islam*edu5cat + relax*edu5cat + tighten*edu5cat, 
                         design=svydat.tun.muslim, family="quasibinomial")
mod.int.edu.idn = glm(dvalcoholexp ~ islam*edu5cat + relax*edu5cat + tighten*edu5cat, 
                      data=dat.idn.muslim, family=binomial(link="logit")) 


### Continuous education with covariates
cov.int.edu.tun = svyglm(dvalcoholexp ~ female + agecat + income + polinterest + shalat + socialwelfare +
                           islam*edu5cat + relax*edu5cat + tighten*edu5cat, 
                         design=svydat.tun.muslim, family="quasibinomial")
cov.int.edu.idn = glm(dvalcoholexp ~ female + agecat + income + polinterest + shalat + socialwelfare + 
                        islam*edu5cat + relax*edu5cat + tighten*edu5cat, 
                      data = dat.idn.muslim, family=binomial(link="logit"))


### Interaction with income
mod.int.income.tun = svyglm(dvalcoholexp ~ islam*income + relax*income + tighten*income, 
                         design=svydat.tun.muslim, family="quasibinomial")
mod.int.income.idn = glm(dvalcoholexp ~ islam*income + relax*income + tighten*income, 
                      data=dat.idn.muslim, family=binomial(link="logit")) 



### Interaction with college
mod.int.college.tun = svyglm(dvalcoholexp ~ islam*college + relax*college + tighten*college, 
                                design=svydat.tun.muslim, family="quasibinomial")
mod.int.college.idn = glm(dvalcoholexp ~ islam*college + relax*college + tighten*college, 
                             data=dat.idn.muslim, family=binomial(link="logit"))


####################################################################
## Out Regression Table
####################################################################

### Regression Table in Main Text ###########################################

tab_model(mod.basic.idn, mod.basic.tun,
          mod.int.highschool.idn, mod.int.highschool.tun,
          mod.int.idn, mod.int.tun, 
          
          show.se = TRUE, collapse.se = TRUE,
          digits = 3,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE, transform = NULL,
          file="Table 2.html")


### Online Appendix: With Covariates ###########################################

tab_model(cov.basic.idn, cov.basic.tun,
          cov.int.highschool.idn, cov.int.highschool.tun,
          cov.int.idn, cov.int.tun, 
          
          show.se = TRUE, collapse.se = TRUE,
          digits = 3,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE, transform = NULL,
          file="Table A5.html")


### Online Appendix: Continuous Measure of Education ###########################################

tab_model(mod.int.edu.idn, mod.int.edu.tun,
          cov.int.edu.idn, cov.int.edu.tun,
          
          show.se = TRUE, collapse.se = TRUE,
          digits = 3,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE, transform = NULL,
          file="Table A6.html")


### Online Appendix: Interaction with Income ###########################################

tab_model(mod.int.income.idn, mod.int.income.tun,
          
          show.se = TRUE, collapse.se = TRUE,
          digits = 3,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE, transform = NULL,
          file="Table A7.html")


### Online Appendix: Interaction with College ###########################################

tab_model(mod.int.college.idn, mod.int.college.tun,
          
          show.se = TRUE, collapse.se = TRUE,
          digits = 3,
          p.val = "wald",
          p.style = "stars",
          show.ci = FALSE, show.p = FALSE, transform = NULL,
          file="Table A8.html")


## Plot Interaction Effects with College

### Tunisia #################
out = summary(margins(mod.int.college.tun, at = list(college=c(0, 1)), design=svydat.tun.muslim))[-c(1,2), 1:4]
colnames(out) = c("Treatment", "Education", "Estimate", "se")
out$Education = c("No College", "College")
out$Education = factor(out$Education, levels = c("No College", "College"))

out$Treatment = c("Islamist", "Islamist",
                  "Liberal\nInitiative", "Liberal\nInitiative", 
                  "Conservative\nInitiative", "Conservative\nInitiative")
out$Treatment = factor(out$Treatment,
                       levels = c("Islamist", 
                                  "Liberal\nInitiative", 
                                  "Conservative\nInitiative"))
plot.temp = ggplot(out, aes(Treatment, Estimate))
plot.int.college.tun = plot.int2(plot.temp, 
                                    title="Marginal Effects of the Treatments by Education (TUN)",
                                    obj=out$Education,
                                    obj.label = "Education",
                                    pos.legend = c(.20, .10),
                                    legend.title.size = 8,
                                    legend.label.size = 8,
                                    lab.y = "Estimate",
                                    ylim = c(-.9, .2),
                                    size.x = 10,
                                    size.y = 10,
                                    title.size = 10) +
  geom_hline(yintercept=0, linetype="dashed") +
  coord_flip()

png("Figure A8.png", width=3000, height=2000, res=300)
plot.int.college.tun
dev.off()

